.check.alpha <-
function (call, alpha, n.stud) 
{
if (!is.numeric(alpha)) {
.stop(call, "alpha must be a numeric vector")
}
if (any(is.na(alpha))) {
.stop(call, "alpha cannot have missing values")
}
if (any(alpha <= 0) || any(alpha >= 1)) {
.stop(call, "alpha cannot be <= 0 or >= 1")
}
if (length(alpha) == 1) {
return(rep(alpha, n.stud))
}
if (length(alpha) != n.stud) {
.stop(call, "alpha has an incorrect length")
}
alpha
}
.check.formula <-
function (call, formula, n.stud) 
{
if (!inherits(formula, "formula")) {
.stop(call, "formula must be a formula")
}
terms <- terms(formula)
xnames <- attr(terms, "term.labels")
if (length(xnames) == 0) {
return(list(formula = "~ 1", matrix = matrix(1, n.stud), 
labels = "(Mean)"))
}
formula <- paste("~", paste(xnames, collapse = " + "))
if (!attr(terms, "intercept")) {
warning("You have specified a regression though the origin")
formula <- paste(formula, "- 1")
}
current.na.action <- options("na.action")$na.action
options(na.action = "na.pass")
X <- model.matrix(as.formula(formula), parent.frame(2))
options(na.action = current.na.action)
if (nrow(X) != n.stud) {
.stop(call, "Independent variables of the formula have an incorrect length")
}
if (any(is.na(X))) {
.stop(call, "Independent variables of the formula cannot have missing values. Impute missing values (using for example the R package 'mi'), call 'meta' for each imputation, and combine all imputations")
}
list(formula = formula, matrix = matrix(c(X), n.stud), labels = colnames(X))
}
.check.hypothesis <-
function (call, hypothesis, model) 
{
n.coef <- ncol(model$matrix)
labels <- model$labels
if (is.null(hypothesis)) {
if (n.coef == 1) {
hypothesis = list(text = paste(labels[1], "=0", sep = ""), 
matrix = matrix(1))
}
else {
hypothesis = list(text = paste(labels[2], "=0", sep = ""), 
matrix = matrix(c(0, 1, rep(0, n.coef - 2)), 
1))
}
}
else if (is.matrix(hypothesis)) {
if (ncol(hypothesis) != n.coef) {
.stop(call, "Wrong number of columns in the hypothesis")
}
text = c()
for (i in 1:nrow(hypothesis)) {
text_i = ""
for (j in 1:ncol(hypothesis)) {
hypothesis_j = hypothesis[i, j]
if (hypothesis_j != 0) {
if (hypothesis_j > 0 && nchar(text_i) > 0) {
text_i = paste(text_i, "+", sep = "")
}
else if (hypothesis_j < 0) {
text_i = paste(text_i, "-", sep = "")
}
if (abs(hypothesis_j != 1)) {
text_i = paste(text_i, hypothesis_j, "*", 
sep = "")
}
text_i = paste(text_i, labels[j], sep = "")
}
}
text = c(text, paste(text_i, "=0", sep = ""))
}
if (nrow(hypothesis) > 1) {
warning("All rows of the hypothesis are given the same weight in the MLE step; please adjust if required.")
}
hypothesis = list(text = text, matrix = hypothesis)
}
else if (is.numeric(hypothesis)) {
if (length(hypothesis) != n.coef) {
.stop(call, "Wrong vector length in the hypothesis")
}
text = ""
for (j in 1:length(hypothesis)) {
hypothesis_j = hypothesis[j]
if (hypothesis_j != 0) {
if (hypothesis_j > 0 && nchar(text) > 0) {
text = paste(text, "+", sep = "")
}
else if (hypothesis_j < 0) {
text = paste(text, "-", sep = "")
}
if (abs(hypothesis_j != 1)) {
text = paste(text, hypothesis_j, "*", sep = "")
}
text = paste(text, labels[j], sep = "")
}
}
hypothesis = list(text = paste(text, "=0", sep = ""), 
matrix = matrix(hypothesis, 1))
}
else {
.stop(call, "Numeric vector or matrix expected in the hypothesis")
}
hypothesis
}
.check.labels <-
function (call, labels, n.stud) 
{
if (!is.vector(labels) && !is.factor(labels)) {
.stop(call, "labels must be a vector")
}
if (length(labels) == 1) {
return(paste0(labels, 1:n.stud))
}
if (length(labels) != n.stud) {
.stop(call, "labels has an incorrect length")
}
as.character(labels)
}
.check.n <-
function (call, n, min.n, n.stud) 
{
if (!is.numeric(n)) {
.stop(call, "n must be a numeric vector")
}
if (any(is.na(n))) {
.stop(call, "n cannot have missing values")
}
if (any(n < min.n)) {
.stop(call, paste("n cannot be <", min.n))
}
if (length(n) == 1) {
return(rep(n, n.stud))
}
if (length(n) != n.stud) {
.stop(call, "n has an incorrect length")
}
n
}
.d_j <-
function (x) 
{
j <- ifelse(x < 1, -1, 1) * exp(lgamma(x/2) - 0.5 * log(x/2) - 
lgamma((x - 1)/2))
na.j <- which(is.na(j))
j[na.j] <- 1 - 3/(4 * x[na.j] - 1)
j
}
.elliptic.q <-
function (x, y, p = 0.95, col = "#cccccc", segments = 51) 
{
center <- c(mean(x), mean(y))
shape <- cov(cbind(x, y))
radius <- sqrt(2 * qf(p, 2, length(x) - 1))
angles <- (0:segments) * 2 * pi/segments
circle <- cbind(cos(angles), sin(angles))
choleski <- chol(shape, pivot = TRUE)
polygon(t(center + radius * t(circle %*% choleski[, order(attr(choleski, 
"pivot"))])), col = col, border = NA)
}
.estimate_n.mle_discard <-
function (N) 
{
lof = 1
while (N > switch(as.character(lof), `1` = 30, `2` = 59, 
`3` = 77, 34 + lof * 15)) {
lof = lof + 1
}
lof
}
.format.0pos <-
function (x) 
{
formatC(x, 0, width = 2, format = "f")
}
.format.1 <-
function (x) 
{
formatC(x, 1, width = 4, format = "f")
}
.format.1pos <-
function (x) 
{
formatC(x, 1, width = 3, format = "f")
}
.format.2 <-
function (x) 
{
formatC(x, 2, width = 5, format = "f")
}
.format.2pos <-
function (x) 
{
formatC(x, 2, width = 4, format = "f")
}
.format.3 <-
function (x) 
{
formatC(x, 3, width = 6, format = "f")
}
.format.4 <-
function (x) 
{
formatC(x, 4, width = 7, format = "f")
}
.format.4pos <-
function (x) 
{
formatC(x, 4, width = 6, format = "f")
}
.format.measure <-
function (measure) 
{
switch(measure, cor = "correlation", `cor in smd` = "correlation in standardized mean difference", 
smc = "standardized mean change", smd = "standardized mean difference")
}
.format.perc2 <-
function (x) 
{
formatC(100 * x, 2, width = 5, format = "f")
}
.format.prob <-
function (p) 
{
p <- formatC(p, digits = 4, format = "f")
p[p == "0.0000"] <- "<.0001"
p
}
.format.prob.html <-
function (p) 
{
p <- formatC(p, digits = 4, format = "f")
p[p == "0.0000"] <- "&lt;0.0001"
p
}
.format.sign <-
function (p) 
{
symnum(p, cutpoints = c(0, 0.001, 0.01, 0.05, 0.1, 1), symbols = c("***", 
"**", "*", ".", " "), na = FALSE, corr = FALSE)
}
.gui <-
function () 
{
.require("base64enc")
.args <- commandArgs(trailingOnly = TRUE)
.output <- .args[1]
.input <- .args[2]
.X <- tryCatch(read.table(.input, header = TRUE), warning = function(w) {
w
}, error = function(e) {
e
})
if (!is.data.frame(.X)) {
stop(paste("Could not open the file", .input))
}
.y <- .X[, match(.args[4], colnames(.X))]
.nsue <- switch(.args[3], z_from_r = smc_from_t(.y, .X$n1, 
labels = .X$study), smc_from_t = smc_from_t(.y, .X$n1, 
labels = .X$study), smd_from_t = smd_from_t(.y, .X$n1, 
.X$n2, labels = .X$study))
if (.args[5] == "lm") {
attach(.X)
.m <- meta(.nsue, as.formula(paste("~", paste(strsplit(.args[6], 
",")[[1]], collapse = "+"))), as.numeric(strsplit(.args[7], 
",")[[1]]))
detach(.X)
}
else {
.m <- meta(.nsue)
}
.measure <- .m$measure
sink(paste(dirname(.input), "/", .output, ".html", sep = ""))
cat("<!DOCTYPE html><html lang=\"en\"><head><meta charset=\"utf-8\" /><title>MetaNSUE -", 
.output, "</title><style type=\"text/css\">html{background-color:#f9f9f9}body{background-color:#fff;border:1px solid #ccc;border-radius:10px;font-family:Arial,\"Trebuchet MS\";font-size:1em;padding:15px}a{color:#36c;text-decoration:none}a:visited{color:#47d}a:hover{color:#c36}em{font-size:110%}h1{margin:0}h2{color:#26527D;font-size:1.5em;font-weight:400;margin-top:2em;text-transform:uppercase}img{border:2px dotted red}li{line-height:150%}table{border-spacing:0;border-top:1px solid black;border-bottom:1px solid black}th{font-weight: normal;border-bottom:1px solid black;padding:10px;text-align:center}td{padding:10px;text-align:center}</style><body>")
cat("<table style=\"width:100%\"><tr>")
cat("<td style=\"text-align:left\"><h1>", .output, "</h1></td>")
cat("<td style=\"text-align:right\"><a href=\"http://www.sdmproject.com/\"><img src=\"data:image/png;base64,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\" alt=\"SDM Project\" style=\"border:0\" /></a>")
cat("</tr></table>")
cat("<ul>")
cat("<li>Measure:<i>", switch(.measure, cor = "correlation", 
smc = "standardized mean change", smd = "standardized mean difference"), 
"</i></li>")
cat("<li>Known measures:", length(.m$known$i))
if (length(.m$known$i) == 0) {
cat(" <span style=\"color:red\"><b>&larr;</b> WARNING: No known measures!</span>")
}
cat("</li>")
cat("<li>Non statistically-significant unknown measures:", 
length(.m$unknown$i), "</li>")
if (nrow(.m$rm$M) < length(.m$known$i) + length(.m$unknown$i)) {
cat("<li>Measures after combination of repeated-measures:", 
nrow(.m$rm$M), "</li>")
}
cat("<li>Imputations:", ncol(.m$unknown$y), "</li>")
cat("<li>Model: <i>measure", .m$model$formula, "</i></li>")
cat("<li>Hypothesis: <i>", paste(.m$hypothesis$text, collapse = " & "), 
"</i></li>")
cat("</ul>")
cat("<h2>Hypothesis test</h2>")
cat("<table>")
cat("<tr><th></th>")
if (.measure == "cor") {
cat("<th>Corr</th>")
}
else {
cat("<th>Estimate</th>")
}
cat("<th><i>Z</i> statistic</th><th><i>P</i> value</th><th>CI<sub><i>lower</i></sub></th><th>CI<sub><i>upper</i></sub></th><th></th></tr>")
cat("<tr><td>", .m$hypothesis$text, "</td>")
.coef <- .m$hypothesis$coef
if (.measure == "cor") {
cat("<td>", .format.3(tanh(.coef)), "</td>")
}
else {
cat("<td>", .format.4(.coef), "</td>")
}
cat("<td>", .format.4(.m$hypothesis$z), "</td><td>", .format.prob.html(.m$hypothesis$p.value), 
"</td>")
ci.low <- .coef + qnorm(0.025) * .m$hypothesis$se
ci.up <- .coef + qnorm(0.975) * .m$hypothesis$se
if (.measure == "cor") {
cat("<td>", .format.3(tanh(ci.low)), "</td><td>", format.3(tanh(ci.up)), 
"</td>")
}
else {
cat("<td>", .format.4(ci.low), "</td><td>", .format.4(ci.up), 
"</td>")
}
cat("<td>", .format.sign(.m$hypothesis$p.value), "</td></tr>")
cat("</table>")
cat("<h2>Residual heterogeneity</h2>")
cat("<table>")
cat("<tr><th><i>&tau;</i><sup>2</sup></th><th><i>I</i><sup>2</sup></th><th><i>H</i><sup>2</sup></th><th><i>F</i> statistic</th><th>df<sub>1</sub></th><th>df<sub>2</sub></th><th><i>P</i> value</th><th></th></tr>")
cat("<tr><td>", .format.4pos(.m$heterogeneity$tau2), "</td><td>", 
.format.2(100 * .m$heterogeneity$i2), "%</td><td>", .format.2pos(.m$heterogeneity$h2), 
"</td><td>", .format.2pos(.m$heterogeneity$q$f), "</td><td>", 
.m$heterogeneity$q$df1, "</td><td>", .format.1pos(.m$heterogeneity$q$df2), 
"</td><td>", .format.prob.html(.m$heterogeneity$q$p.value), 
"</td><td>", .format.sign(.m$heterogeneity$q$p.value), 
"</td></tr>", sep = "")
cat("</table>")
if (.args[5] == "mean" && length(.args) > 5) {
.extra <- strsplit(.args[6], ",")[[1]]
if ("leave1out" %in% .extra) {
cat("<h2>Leave-one-out</h2>")
.jk <- leave1out(.nsue)
cat("<table>")
cat("<tr><th>Discarded study</th>")
if (.measure == "cor") {
cat("<th>Corr</th>")
}
else {
cat("<th>Estimate</th>")
}
cat("<th><i>Z</i> statistic</th><th><i>P</i> value</th><th>CI<sub><i>lower</i></sub></th><th>CI<sub><i>upper</i></sub></th><th></th></tr>")
for (i in 1:length(.jk)) {
.jki <- .jk[[i]]$meta.nsue
cat("<tr><td>", .jk[[i]]$study, "</td>")
.coef <- .jki$hypothesis$coef
if (.measure == "cor") {
cat("<td>", .format.3(tanh(.coef)), "</td>")
}
else {
cat("<td>", .format.4(.coef), "</td>")
}
cat("<td>", .format.4(.jki$hypothesis$z), "</td><td>", 
.format.prob.html(.jki$hypothesis$p.value), 
"</td>")
ci.low <- .coef + qnorm(0.025) * .jki$hypothesis$se
ci.up <- .coef + qnorm(0.975) * .jki$hypothesis$se
if (.measure == "cor") {
cat("<td>", .format.3(tanh(ci.low)), "</td><td>", 
.format.3(tanh(ci.up)), "</td>")
}
else {
cat("<td>", .format.4(ci.low), "</td><td>", 
.format.4(ci.up), "</td>")
}
cat("<td>", .format.sign(.jki$hypothesis$p.value), 
"</td></tr>")
}
cat("</table>")
}
if ("metabias" %in% .extra) {
cat("<h2>Test for funnel plot asymmetry</h2>")
.bias <- metabias(.m)
cat("<table>")
cat("<tr><th><i>Z</i> statistic</th><th><i>P</i> value</th><th></th></tr>")
cat("<tr><td>", .format.4(.bias$statistic), "</td><td>", 
.format.prob.html(.bias$p.value), "</td><td>", 
.format.sign(.bias$p.value), "</td></tr>")
cat("</table>")
}
if ("forest" %in% .extra) {
cat("<h2>Forest plot</h2>")
cat("<!--")
pdf(file = NULL)
.forest <- forest(.m)
dev.off()
png(".tmp", .forest$optimal.width, .forest$optimal.height, 
"in", pointsize = 9, res = 300)
forest(.m)
dev.off()
cat("-->")
img <- dataURI(file = ".tmp", mime = "image/png")
cat("<p>Click the plot to enlarge it:</p>")
cat("<a href=\"", img, "\" target=\"_blank\"><img src=\"", 
img, "\" alt=\"Forest plot\" style=\"width:40%\" /></a>")
}
if ("funnel" %in% .extra) {
cat("<h2>Funnel plot</h2>")
cat("<!--")
png(".tmp", 5, 5, "in", pointsize = 9, res = 300)
funnel(.m)
dev.off()
cat("-->")
img <- dataURI(file = ".tmp", mime = "image/png")
cat("<p>Click the plot to enlarge it:</p>")
cat("<a href=\"", img, "\" target=\"_blank\"><img src=\"", 
img, "\" alt=\"Funnel plot\" style=\"width:40%\" /></a>")
}
}
cat("<hr style=\"margin-top:50px\" />")
cat("<p style=\"text-align:right\"><a href=\"http://www.metansue.com/\">MetaNSUE GUI</a> is part of the <a href=\"http://www.sdmproject.com/\">SDM Project</a>. To conduct other or more personalized analyses use <a href=\"https://cran.r-project.org/web/packages/metansue/i\">MetaNSUE R package</a>.</p>")
cat("</body></html>")
}
.meta.nsue <-
function (x, model, hypothesis, n.imp, n.bins, maxiter, tol) 
{
measure <- x$measure
y <- x$y
x$y = NULL
y_lo = x$y_lo
x$y_lo = NULL
y_up = x$y_up
x$y_up = NULL
n.stud <- length(y)
known <- which(!is.na(y) | y_up - y_lo < 1e-06)
unknown <- setdiff(1:n.stud, known)
y[is.na(y)] = (y_up[is.na(y)] + y_lo[is.na(y)])/2
if (measure == "cor" || measure == "cor in smd") {
y.var = x$y.var
y_lo.var = y.var
y_up.var = y.var
}
if (measure == "smc" || measure == "smd") {
y2var_k1 <- x$y2var_k1
y2var_k2 <- x$y2var_k2
y.var = y2var_k1 + y2var_k2 * y^2
y_lo.var = y2var_k1 + y2var_k2 * y_lo^2
y_up.var = y2var_k1 + y2var_k2 * y_up^2
}
X = model$matrix
n.coef = ncol(X)
if (length(unknown)) {
if (measure == "cor" || measure == "cor in smd") {
mll_coef <- function(coef, known, known.y, known.y.var, 
known.weights, unknown, unknown.y_lo, unknown.y_lo.se, 
unknown.y_up, unknown.y_up.se, unknown.weights, 
X) {
mu <- X %*% coef
unknown.mu <- mu[unknown]
a = pnorm((unknown.y_up - unknown.mu)/unknown.y_up.se, 
log.p = TRUE)
c = pnorm((unknown.y_lo - unknown.mu)/unknown.y_up.se, 
log.p = TRUE)
sum(known.weights * (log(known.y.var) + (known.y - 
mu[known])^2/known.y.var))/2 - sum(unknown.weights * 
(a + log(-expm1(c - a))))
}
mll_tau2 <- function(tau2, known.err2, known.y.var, 
known.weights, unknown.err_lo, unknown.y_lo.var, 
unknown.err_up, unknown.y_up.var, unknown.weights) {
if (tau2 < 0) {
return(Inf)
}
known.sigma2 <- known.y.var + tau2
a = pnorm(unknown.err_up/sqrt(unknown.y_up.var + 
tau2), log.p = TRUE)
c = pnorm(unknown.err_lo/sqrt(unknown.y_lo.var + 
tau2), log.p = TRUE)
sum(known.weights * (log(known.sigma2) + known.err2/known.sigma2))/2 - 
sum(unknown.weights * (a + log(-expm1(c - a))))
}
mi1 <- function(tau2, mu, y_lo, y_up, y.var, rm.var, 
rm.y, n.imp) {
sigma2 <- y.var + tau2
sigma <- sqrt(sigma2)
if (length(rm.var)) {
rm.sigma <- sqrt(rm.var + tau2)
Sxx <- rm.sigma %*% t(rm.sigma) * (diag(1 - 
rm$r, length(rm.var)) + rm$r)
Sxy <- rm.sigma * sigma * rm$r
beta <- solve(Sxx) %*% Sxy
mus <- rm.y %*% beta
sigma <- sqrt(sigma2 - t(beta) %*% Sxx %*% 
beta)
}
else {
mus <- rep(mu, n.imp)
}
q <- rep(NA, n.imp)
to_imp <- 1:n.imp
while (length(to_imp)) {
q[to_imp] <- rnorm(length(to_imp), mus[to_imp], 
sigma)
to_imp <- which(q <= y_lo & q >= y_up)
}
q
}
}
if (measure == "smc" || measure == "smd") {
mll_coef <- function(coef, known, known.y, known.y.var, 
known.weights, unknown, unknown.y_lo, unknown.y_lo.se, 
unknown.y_up, unknown.y_up.se, unknown.weights, 
X) {
mu <- X %*% coef
unknown.mu <- mu[unknown]
a = pnorm((unknown.y_up - unknown.mu)/unknown.y_up.se, 
log.p = TRUE)
c = pnorm((unknown.y_lo - unknown.mu)/unknown.y_lo.se, 
log.p = TRUE)
sum(known.weights * (log(known.y.var) + (known.y - 
mu[known])^2/known.y.var))/2 - sum(unknown.weights * 
(a + log(-expm1(c - a))))
}
mll_tau2 <- function(tau2, known.err2, known.y.var, 
known.weights, unknown.err_lo, unknown.y_lo.var, 
unknown.err_up, unknown.y_up.var, unknown.weights) {
if (tau2 < 0) {
return(Inf)
}
known.sigma2 <- known.y.var + tau2
a = pnorm(unknown.err_up/sqrt(unknown.y_up.var + 
tau2), log.p = TRUE)
c = pnorm(unknown.err_lo/sqrt(unknown.y_lo.var + 
tau2), log.p = TRUE)
sum(known.weights * (log(known.sigma2) + known.err2/known.sigma2))/2 - 
sum(unknown.weights * (a + log(-expm1(c - a))))
}
mi2 <- function(tau2, mu, y_lo, y_up, y2var_k1, y2var_k2, 
rm.var_k1, rm.var_k2, rm.y, n.imp, n.bins) {
sigma2 <- y2var_k1 + y2var_k2 * mu^2 + tau2
sigma <- sqrt(sigma2)
if (length(rm.var_k1)) {
rm.sigma <- sqrt(rm.var_k1 + rm.var_k2 * mu^2 + 
tau2)
Sxx <- rm.sigma %*% t(rm.sigma) * (diag(1 - 
rm$r, length(rm.var_k1)) + rm$r)
Sxy <- rm.sigma * sigma * rm$r
beta <- solve(Sxx) %*% Sxy
mus <- rm.y %*% beta
sigma <- sqrt(sigma2 - t(beta) %*% Sxx %*% 
beta)
}
else {
mus <- rep(mu, n.imp)
}
width <- (y_up - y_lo)/n.bins
y <- sample(seq(y_lo + width/2, y_up - width/2, 
width))
q <- c()
for (imp in 1:n.imp) {
raw_dens <- dnorm(y, mus[imp], sigma) * (y2var_k1 + 
y2var_k2 * y^2 + tau2)
pfun <- cumsum(raw_dens/sum(raw_dens))
p <- runif(1)
j <- 1
while (p > pfun[j]) {
j <- j + 1
}
q <- c(q, y[j])
}
q
}
}
}
rm <- x$rm
rm.M <- t(unname(model.matrix(~0 + x$labels)))
rm.M <- rm.M[unique(c(1:nrow(rm.M) %*% rm.M)), ]
rm_weights = apply(apply(rm.M, 1, function(x) {
x/(1 + (sum(x) - 1) * rm$r)
}), 1, sum)
rm$M <- rm.M
rm$weights <- rm_weights
x$rm = rm
if (length(unknown)) {
coef = NULL
n.mle_discard = .estimate_n.mle_discard(n.stud)
mle_discard = c()
while (length(mle_discard) < n.mle_discard) {
min_hyp = Inf
for (i in setdiff(1:n.stud, mle_discard)) {
sample_i = (1:n.stud)[-c(mle_discard, i)]
known_i = sample_i[sample_i %in% known]
unknown_i = sample_i[sample_i %in% unknown]
if (n.coef == 1) {
interval = c(min(c(y[known_i], y_lo[unknown_i])), 
max(c(y[known_i], y_up[unknown_i])))
coef_i <- optimize(mll_coef, interval, known_i, 
y[known_i], y.var[known_i], rm_weights[known_i], 
unknown_i, y_lo[unknown_i], sqrt(y_lo.var[unknown_i]), 
y_up[unknown_i], sqrt(y_up.var[unknown_i]), 
rm_weights[unknown_i], X)$minimum
}
else {
initial_coef <- coef(lm.wfit(X[sample_i, ], 
y[sample_i], 1/y.var[sample_i]))
if (measure == "cor" || measure == "cor in smd") {
coef_i <- optim(initial_coef, mll_coef, gr = NULL, 
known_i, y[known_i], y.var[known_i], rm_weights[known_i], 
unknown_i, y_lo[unknown_i], sqrt(y_lo.var[unknown_i]), 
y_up[unknown_i], sqrt(y_up.var[unknown_i]), 
rm_weights[unknown_i], X)$par
}
if (measure == "smc" || measure == "smd") {
coef_i <- optim(initial_coef, mll_coef, gr = NULL, 
known_i, y[known_i], y.var[known_i], rm_weights[known_i], 
unknown_i, y_lo[unknown_i], sqrt(y_lo.var[unknown_i]), 
y_up[unknown_i], sqrt(y_up.var[unknown_i]), 
rm_weights[unknown_i], X)$par
}
}
hyp_i = sum(abs(hypothesis$matrix %*% coef_i))
if (hyp_i < abs(min_hyp)) {
min_hyp = hyp_i
min_i = i
min_coef = coef_i
}
}
mle_discard = c(mle_discard, min_i)
}
mu = X %*% min_coef
tau2 <- optimize(mll_tau2, c(0, 999), (y[known] - mu[known])^2, 
y.var[known], rm_weights[known], y_lo[unknown] - 
mu[unknown], y_lo.var[unknown], y_up[unknown] - 
mu[unknown], y_up.var[unknown], rm_weights[unknown])$minimum
mi_y <- NULL
for (i in unknown) {
rm.indexs <- which(x$labels == x$labels[i] & 1:n.stud < 
i)
if (measure == "cor" || measure == "cor in smd") {
rm.var <- c()
}
if (measure == "smc" || measure == "smd") {
rm.var_k1 <- c()
rm.var_k2 <- c()
}
rm.y = NULL
for (j in rm.indexs) {
is.known = !is.na(y[j])
if (measure == "cor" || measure == "cor in smd") {
rm.var <- c(rm.var, y.var[j])
}
if (measure == "smc" || measure == "smd") {
rm.var_k1 <- c(rm.var_k1, y2var_k1[j])
rm.var_k2 <- c(rm.var_k2, y2var_k2[j])
}
if (is.known) {
rm.y <- cbind(rm.y, rep(y[j], n.imp))
}
else {
rm.y <- cbind(rm.y, mi_y[match(j, unknown), 
])
}
}
if (measure == "cor" || measure == "cor in smd") {
mi_y <- rbind(mi_y, mi1(tau2, mu[i], y_lo[i], 
y_up[i], y.var[i], rm.var, rm.y, n.imp))
}
if (measure == "smc" || measure == "smd") {
mi_y <- rbind(mi_y, mi2(tau2, mu[i], y_lo[i], 
y_up[i], y2var_k1[i], y2var_k2[i], rm.var_k1, 
rm.var_k2, rm.y, n.imp, n.bins))
}
}
colnames(mi_y) <- NULL
}
else {
mi_y = matrix(nrow = 0, ncol = 0)
}
x$known = list(i = known, y = y[known])
x$unknown = list(i = unknown, y = mi_y)
class(x) <- "meta.nsue"
.meta.nsue2(x, model, hypothesis, maxiter, tol)
}
.meta.nsue2 <-
function (x, model, hypothesis, maxiter, tol) 
{
measure = x$measure
known = x$known$i
unknown = x$unknown$i
y = rep(NA, length(known) + length(unknown))
y[known] = x$known$y
mi_y = x$unknown$y
if (measure == "cor" || measure == "cor in smd") {
y.var = x$y.var
}
if (measure == "smc" || measure == "smd") {
y2var_k1 <- x$y2var_k1
y2var_k2 <- x$y2var_k2
}
rm = x$rm
rm.M = rm$M
rm_weights = rm$weights
X = model$matrix
n.coef = ncol(X)
df <- nrow(rm.M) - n.coef
rm.M2 <- t(apply(rm.M, 1, function(x) {
x/sum(x)
}))
if (measure == "cor" || measure == "cor in smd") {
y.var <- y.var/rm_weights
ay.var <- apply(rm.M, 1, function(xx) {
mean(y.var[which(xx == 1)])/sum(xx)
})
}
if (measure == "smc" || measure == "smd") {
y2var_k1 <- y2var_k1/rm_weights
ay2var_k1 <- apply(rm.M, 1, function(xx) {
mean(y2var_k1[which(xx == 1)])/sum(xx)
})
y2var_k2 <- y2var_k2
ay2var_k2 <- apply(rm.M, 1, function(xx) {
mean(y2var_k2[which(xx == 1)])
})
}
aX <- rm.M2 %*% X
mi.coef <- NULL
mi.cov <- NULL
mi.tau2 <- c()
mi.qe <- c()
mi.i2 <- c()
mi.h2 <- c()
for (j in 1:max(c(1, ncol(mi_y)))) {
if (ncol(mi_y) > 0) {
y[unknown] <- mi_y[, j]
}
ay <- c(rm.M2 %*% y)
if (measure == "smc" || measure == "smd") {
ay.var <- ay2var_k1 + ay2var_k2 * ay^2
}
W_fe <- diag(1/ay.var)
P_fe <- W_fe - W_fe %*% aX %*% solve(t(aX) %*% W_fe %*% 
aX) %*% t(aX) %*% W_fe
tau2_j <- .tau2.reml(ay, ay.var, aX, maxiter, tol)
W <- diag(1/(ay.var + tau2_j))
inv_XtWX <- solve(t(aX) %*% W %*% aX)
h2_j <- 1 + tau2_j/df * sum(diag(P_fe))
mi.coef <- cbind(mi.coef, inv_XtWX %*% t(aX) %*% W %*% 
ay)
mi.cov <- cbind(mi.cov, c(inv_XtWX))
mi.tau2 <- c(mi.tau2, tau2_j)
mi.qe <- c(mi.qe, max(0, t(ay) %*% P_fe %*% ay))
mi.i2 <- c(mi.i2, max(0, 1 - 1/h2_j))
mi.h2 <- c(mi.h2, h2_j)
}
coef <- apply(mi.coef, 1, mean)
cov <- .pool.cov(mi.coef, mi.cov)
tau2 = mean(mi.tau2)
f_df2 <- .pool.chi2(mi.qe, df)
f <- f_df2[1]
df2 <- f_df2[2]
i2 = mean(mi.i2)
h2 = mean(mi.h2)
model$coef <- coef
model$cov <- cov
model$se <- sqrt(diag(cov))
x$model = model
x$heterogeneity <- list(tau2 = tau2, h2 = h2, i2 = i2, q = data.frame(f, 
df1 = df, df2, p.value = 1 - pf(f, df, df2)))
h <- hypothesis$matrix
hcoef <- c(h %*% coef)
hcov <- h %*% cov %*% t(h)
hypothesis$coef <- hcoef
if (nrow(h) == 1) {
hse <- sqrt(hcov)
z <- hcoef/hse
hypothesis$se <- hse
hypothesis$z <- z
hypothesis$p.value <- 2 * pnorm(-abs(z))
}
else {
qr <- qr(hcov)
pivot <- qr$pivot[1:qr$rank]
chisq <- c(t(hcoef[pivot]) %*% solve(hcov[pivot, pivot]) %*% 
hcoef[pivot])
df <- length(pivot)
hypothesis$chisq <- chisq
hypothesis$df <- df
hypothesis$p.value <- 1 - pchisq(chisq, df)
}
x$hypothesis = hypothesis
x
}
.pool.chi2 <-
function (chi2, df1) 
{
m <- length(chi2)
if (m == 1) {
return(c(chi2/df1, Inf))
}
r <- (1 + 1/m) * var(sqrt(chi2))
c((mean(chi2)/df1 - (m + 1)/(m - 1) * r)/(1 + r), (m - 1)/df1^(3/m) * 
(1 + 1/r)^2)
}
.pool.cov <-
function (x, x_cov) 
{
m <- ncol(x)
if (m == 1) {
return(matrix(x_cov, nrow(x)))
}
cov0 <- matrix(apply(x_cov, 1, mean), nrow(x))
var0 <- diag(cov0)
var.increase <- sqrt((var0 + (1 + 1/m) * apply(x, 1, var))/var0)
var.increase %*% t(var.increase) * cov0
}
.pool.var <-
function (x, x_var) 
{
if (is.vector(x)) {
m <- length(x)
if (m == 1) {
return(x_var)
}
return(mean(x_var) + (1 + 1/length(x)) * var(x))
}
m <- ncol(x)
if (m == 1) {
return(x_var)
}
apply(x_var, 1, mean) + (1 + 1/ncol(x)) * apply(x, 1, var)
}
.print.heterogeneity <-
function (x) 
{
cat("Residual heterogeneity (tau^2):", .format.4pos(x$heterogeneity$tau2), 
" I^2:", paste(.format.perc2(x$heterogeneity$i2), "%", 
sep = ""), " H^2:", .format.2pos(x$heterogeneity$h2), 
"\n")
cat("F-statistic (heterogeneity):", .format.2pos(x$heterogeneity$q$f), 
"on", x$heterogeneity$q$df1, "and", .format.1pos(x$heterogeneity$q$df2), 
"df Pr(>F):", .format.prob(x$heterogeneity$q$p.value), 
"\n")
}
.print.hypothesis <-
function (x) 
{
coef <- x$hypothesis$coef
nrow = length(coef)
p.value <- x$hypothesis$p.value
prob <- .format.prob(p.value)
sign <- .format.sign(p.value)
if (nrow == 1) {
measure <- x$measure
se <- x$hypothesis$se
cat("One-row hypothesis:\n")
ci.low <- coef + qnorm(0.025) * se
ci.up <- coef + qnorm(0.975) * se
if (measure == "cor" || measure == "cor in smd") {
hypothesis <- cbind(cbind(.format.3(tanh(coef)), 
.format.4(x$hypothesis$z), prob, .format.3(tanh(cbind(ci.low, 
ci.up)))), sign)
colnames(hypothesis) <- c("Corr", "z value", "Pr(>|z|)", 
"CI(low)", "CI(up)", "")
}
if (measure == "smc" || measure == "smd") {
hypothesis <- cbind(cbind(.format.4(coef), .format.4(x$hypothesis$z), 
prob, .format.4(cbind(ci.low, ci.up))), sign)
colnames(hypothesis) <- c("Estimate", "z value", 
"Pr(>|z|)", "CI(low)", "CI(up)", "")
}
}
else {
cat("Multi-row hypothesis:\n")
hypothesis <- cbind(.format.4(coef), c(.format.2pos(x$hypothesis$chisq), 
rep("", nrow - 1)), c(.format.0pos(x$hypothesis$df), 
rep("", nrow - 1)), c(prob, rep("", nrow - 1)), c(sign, 
rep("", nrow - 1)))
colnames(hypothesis) <- c("Estimate", "chisq", "df", 
"Pr(>chisq)", "")
}
rownames(hypothesis) <- x$hypothesis$text
print(hypothesis, quote = FALSE, right = TRUE, print.gap = 2)
}
.print.model <-
function (x) 
{
cat("Model:\n")
table <- cbind(.format.4(x$model$coef), .format.4pos(x$model$se))
colnames(table) <- c("Estimate", "Std. Error")
rownames(table) <- x$model$labels
print(table, quote = FALSE, right = TRUE, print.gap = 2)
}
.print.sign <-
function () 
{
cat("---\n")
cat("Signif. codes: 0 '***' 0.001 '**' 0.01 '*' 0.05 '.' 0.1 ' ' 1\n")
}
.r_in_smd_from_sds <-
function (var.diff, df1, var1.sum, sd1.prod, df2, var2.sum, sd2.prod, 
r.min, r.max) 
{
r <- (df1 * (var1.sum - var.diff) + df2 * (var2.sum - var.diff))/(2 * 
(df1 * sd1.prod + df2 * sd2.prod))
r[r < r.min] = r.min
r[r > r.max] = r.max
r
}
.r_in_smd_from_t_means_and_sds2 <-
function (x, model, hypothesis, maxiter, tol) 
{
x$measure <- "smd"
smd <- x$smd[x$known$i, ]
x$known$y <- smd$j * smd$diff/sqrt((smd$df1 * (smd$var1.sum - 
2 * smd$sd1.prod * tanh(x$known$y)) + smd$df2 * (smd$var2.sum - 
2 * smd$sd2.prod * tanh(x$known$y)))/(smd$df1 + smd$df2))
smd <- x$smd[x$unknown$i, ]
if (length(x$unknown$y)) {
x$unknown$y <- smd$j * smd$diff/sqrt((smd$df1 * (smd$var1.sum - 
2 * smd$sd1.prod * tanh(x$unknown$y)) + smd$df2 * 
(smd$var2.sum - 2 * smd$sd2.prod * tanh(x$unknown$y)))/(smd$df1 + 
smd$df2))
}
x$y2var_k1 = x$smd$y2var_k1
x$y2var_k2 = x$smd$y2var_k2
x$y.var <- NULL
x$smd <- NULL
.meta.nsue2(x, model, hypothesis, maxiter, tol)
}
.require <-
function (package) 
{
if (!require(package, character.only = TRUE, quietly = TRUE)) {
install.packages(package, repos = "https://cloud.r-project.org")
library(package, character.only = TRUE, quietly = TRUE)
}
}
.signif.up <-
function (x, digits = 6) 
{
power <- 10^(round(digits) - 1 - floor(log10(abs(x))))
ceiling(x * power)/power
}
.stop <-
function (call, message) 
{
cat("\n")
print(call)
stop(paste0(message, "\n "), call. = FALSE)
}
.tau2.reml <-
function (y, y.var, X, maxiter, tol) 
{
tau2 = var(y) - mean(y.var)
if (tau2 < 0) {
tau2 = 0
}
for (iter in 1:maxiter) {
old_tau2 <- tau2
W <- diag(1/(y.var + tau2))
P <- W - W %*% X %*% solve(t(X) %*% W %*% X) %*% t(X) %*% 
W
tau2 <- max(0, tau2 + solve(sum(diag(P %*% P))) %*% (t(y) %*% 
P %*% P %*% y - sum(diag((P)))))
if (abs(tau2 - old_tau2) < tol) {
break
}
}
tau2
}
.warning <-
function (message) 
{
warning(message, call. = FALSE)
}
coef.meta.nsue <-
function (object, ...) 
{
table <- cbind(object$model$coef, object$model$se)
colnames(table) <- c("Estimate", "Std. Error")
table
}
fitted.meta.nsue <-
function (object, ...) 
{
object$model$matrix %*% object$model$coef
}
forest <-
function (x, ...) 
UseMethod("forest")
forest.meta.nsue <-
function (x, ...) 
{
if (!inherits(x, "meta.nsue")) {
.stop(match.call(), "The argument must be a 'meta.nsue' object")
}
measure = x$measure
known = x$known$i
unknown = x$unknown$i
unknown.n.stud <- length(unknown)
n.stud <- length(known) + unknown.n.stud
if (nrow(x$rm$M) < n.stud) {
.warning("This plot shows repeated measures as separate studies")
}
if (length(x$hypothesis$coef) > 1) {
.warning("This plot only shows the first row of the hypothesis")
}
labels <- c(x$labels[unknown], x$labels[known], x$hypothesis$text[1])
pos.y <- c(n.stud + 2 - c(unknown, known), 0)
if (unknown.n.stud) {
y <- apply(x$unknown$y, 1, mean)
y.low <- apply(x$unknown$y, 1, function(x) {
quantile(x, 0.025)
})
y.upp <- apply(x$unknown$y, 1, function(x) {
quantile(x, 0.975)
})
if (measure == "cor" || measure == "cor in smd") {
y.se <- sqrt(x$y.var[unknown])
}
if (measure == "smc" || measure == "smd") {
y.se <- sqrt(.pool.var(x$unknown$y, x$y2var_k1[unknown] + 
x$y2var_k2[unknown] * x$unknown$y^2))
}
}
else {
y <- y.low <- y.upp <- y.se <- c()
}
y <- c(y, x$known$y, x$hypothesis$coef[1])
if (measure == "cor" || measure == "cor in smd") {
y.se <- c(y.se, sqrt(x$y.var[known]), x$hypothesis$se[1])
}
if (measure == "smc" || measure == "smd") {
y.se <- c(y.se, sqrt(x$y2var_k1[known] + x$y2var_k2[known] * 
x$known$y^2), x$hypothesis$se[1])
}
ci.low <- y + qnorm(0.025) * y.se
ci.upp <- y + qnorm(0.975) * y.se
if (measure == "cor" || measure == "cor in smd") {
y <- tanh(y)
if (unknown.n.stud) {
y.low <- tanh(y.low)
y.upp <- tanh(y.upp)
}
ci.low <- tanh(ci.low)
ci.upp <- tanh(ci.upp)
}
lwd <- 1/y.se
lwd <- sqrt(9 + 216 * (lwd - min(lwd))/(max(lwd) - min(lwd)))
ci.text <- paste0(.format.2(y), " [ ", .format.2(ci.low), 
", ", .format.2(ci.upp), " ] ", .format.sign(2 * pnorm(-abs(y/y.se))))
plot.new()
xlim <- c(-2.5 - max(strwidth(labels, units = "inches")), 
max(strwidth(ci.text, units = "inches")) + 2.5)
ylim <- c(-2, n.stud + 1)
plot.window(xlim = xlim, ylim = ylim)
xthr <- .signif.up(max(abs(c(quantile(ci.low, 0.1), quantile(ci.upp, 
0.9)))), 1)
lines(rep(0, 2), c(n.stud + 1.5, -1), col = "#bbbbbb", lty = 1)
lines(c(-2, 2), rep(-1, 2), col = "#bbbbbb", lty = 1)
for (pos.x in -2:2) {
lines(rep(pos.x, 2), c(-1, -1.3), col = "#bbbbbb", lty = 1)
text(pos.x, -1.5, .format.2(pos.x/2 * xthr), pos = 1, 
col = "#bbbbbb")
}
for (i in 1:(n.stud + 1)) {
pos.y_i <- pos.y[i]
y_i <- y[i]
ci.low_i <- ci.low[i]
ci.upp_i <- ci.upp[i]
if (i > unknown.n.stud) {
col <- "#000000"
}
else {
y.low_i <- y.low[i]
y.upp_i <- y.upp[i]
if (y.upp_i > -xthr && y.low_i < xthr) {
lines(c(max(y.low_i/xthr * 2, -2), min(y.upp_i/xthr * 
2, 2)), rep(pos.y_i, 2), lwd = lwd[i], col = "#dddddd")
}
col <- "#a7a7a7"
}
if (y_i > -xthr && y_i < xthr) {
lines(rep(y_i/xthr * 2, 2), rep(pos.y_i, 2), lwd = lwd[i], 
col = col)
}
if (ci.upp_i > -xthr && ci.low_i < xthr) {
lines(c(max(ci.low_i/xthr * 2, -2), min(ci.upp_i/xthr * 
2, 2)), rep(pos.y_i, 2), lend = 2, col = col)
if (ci.low_i > -xthr) {
lines(rep(ci.low_i/xthr * 2, 2), pos.y_i + c(-0.15, 
0.15), lend = 2, col = col)
}
if (ci.upp_i < xthr) {
lines(rep(ci.upp_i/xthr * 2, 2), pos.y_i + c(-0.15, 
0.15), lend = 2, col = col)
}
}
text(-2.1, pos.y_i, labels[i], pos = 2, col = col)
text(2.1, pos.y_i, ci.text[i], pos = 4, col = col)
}
width <- round(diff(xlim))
height <- round(diff(ylim)/3)
cat("\n")
cat("Use pdf(filename, width, height) before calling forest to save it.\n")
cat("The optimal width and height of this plot is ~", width, 
" x ~", height, " inches.\n", sep = "")
cat("\n")
invisible(list(optimal.width = width, optimal.height = height))
}
funnel <-
function (x, ...) 
UseMethod("funnel")
funnel.meta.nsue <-
function (x, ...) 
{
if (!inherits(x, "meta.nsue")) {
.stop(match.call(), "The argument must be a 'meta.nsue' object")
}
measure = x$measure
known <- x$known$i
known.n.stud <- length(known)
unknown <- x$unknown$i
unknown.n.stud <- length(unknown)
if (nrow(x$rm$M) < known.n.stud + unknown.n.stud) {
.warning("This analysis does not take repeated measures into account")
}
fitted <- fitted(x)
known.res <- x$known$y - fitted[known]
if (measure == "cor" || measure == "cor in smd") {
known.se <- sqrt(x$y.var[known])
}
if (measure == "smc" || measure == "smd") {
known.se <- sqrt(x$y2var_k1[known] + x$y2var_k2[known] * 
x$known$y^2)
}
if (unknown.n.stud) {
unknown.fitted <- fitted[unknown]
unknown.res <- apply(x$unknown$y, 1, mean) - unknown.fitted
if (measure == "cor" || measure == "cor in smd") {
unknown.se <- sqrt(x$y.var[unknown])
}
if (measure == "smc" || measure == "smd") {
unknown.se <- sqrt(apply(x$y2var_k1[unknown] + x$y2var_k2[unknown] * 
x$unknown$y^2, 1, mean))
}
max.se <- .signif.up(max(c(known.se, unknown.se)), 1)
}
else {
max.se <- .signif.up(max(known.se), 1)
}
ci <- qnorm(0.975) * max.se
plot(NA, NA, type = "n", xlim = 1.3 * c(-ci, ci), ylim = c(max.se, 
0), lty = 2, frame.plot = FALSE, xlab = "Residual effect size", 
ylab = "Standard error")
ci.x <- c(-ci, 0, ci)
ci.y <- c(max.se, 0, max.se)
polygon(c(ci.x, rep(1.3 * ci, 2), rep(-1.3 * ci, 2)), c(ci.y, 
max.se, 0, 0, max.se), col = "#fcfcfc", border = "#dddddd")
if (unknown.n.stud) {
for (i in 1:unknown.n.stud) {
if (measure == "cor" || measure == "cor in smd") {
lines(c(max(quantile(x$unknown$y[i, ] - unknown.fitted[i], 
0.025), -1.3 * ci), min(quantile(x$unknown$y[i, 
] - unknown.fitted[i], 0.975), 1.3 * ci)), 
rep(sqrt(x$y.var[unknown][i]), 2), lwd = 7, 
col = "#dddddd")
}
if (measure == "smc" || measure == "smd") {
.elliptic.q(x$unknown$y[i, ] - unknown.fitted[i], 
sqrt(x$y2var_k1[unknown][i] + x$y2var_k2[unknown][i] * 
x$unknown$y[i, ]^2), col = "#dddddd")
}
}
}
lines(ci.x, ci.y, lty = 2)
lines(c(0, 0), c(max.se, 0), lty = 2)
if (unknown.n.stud) {
for (i in 1:unknown.n.stud) {
lines(rep(unknown.res[i], 2), rep(unknown.se[i], 
2), lwd = 7, col = "#a7a7a7")
}
}
for (i in 1:known.n.stud) {
lines(rep(known.res[i], 2), rep(known.se[i], 2), lwd = 7, 
col = "#000000")
}
cat("\n")
cat("Use pdf(filename) before calling funnel to save it.\n")
cat("\n")
}
leave1out <-
function (x, ...) 
UseMethod("leave1out")
leave1out.nsue <-
function (x, formula = ~1, hypothesis = NULL, n.imp = 500, n.bins = 200, 
maxiter = 200, tol = 1e-06, ...) 
{
call <- match.call()
y <- x$y
measure <- x$measure
n.stud <- length(y)
model <- .check.formula(call, formula, n.stud)
hypothesis <- .check.hypothesis(call, hypothesis, model)
if (n.imp < 2) {
.stop(call, "The number of imputations must be at least 2")
}
if (length(unique(x$labels)) < n.stud) {
.warning("This analysis understand repeated measures as separate studies")
}
nsue_i <- x
model_i <- model
obj <- list()
for (i in 1:n.stud) {
nsue_i$y <- x$y[-i]
nsue_i$y_lo <- x$y_lo[-i]
nsue_i$y_up <- x$y_up[-i]
if (measure == "cor" || measure == "cor in smd") {
nsue_i$y.var <- x$y.var[-i]
}
if (measure == "smc" || measure == "smd") {
nsue_i$y2var_k1 <- x$y2var_k1[-i]
nsue_i$y2var_k2 <- x$y2var_k2[-i]
}
if (measure == "cor in smd") {
nsue_i$smd = x$smd[-i, ]
}
nsue_i$labels <- x$labels[-i]
class(nsue_i) <- "nsue"
model_i$matrix <- as.matrix(model$matrix[-i, ])
obj[[i]] <- list(study = x$labels[i], meta.nsue = .meta.nsue(nsue_i, 
model_i, hypothesis, n.imp, n.bins, maxiter, tol))
}
class(obj) <- "leave1out.nsue"
obj
}
meta <-
function (x, ...) 
UseMethod("meta")
meta.nsue <-
function (x, formula = ~1, hypothesis = NULL, n.imp = 500, n.bins = 200, 
maxiter = 200, tol = 1e-06, ...) 
{
call <- match.call()
if (!inherits(x, "nsue")) {
.stop(call, "Use an smc_from_t, smd_from_t or r_from_z call as the first (nsue) argument.")
}
n.stud <- length(x$y)
model <- .check.formula(call, formula, n.stud)
hypothesis <- .check.hypothesis(call, hypothesis, model)
if (n.imp < 2) {
.stop(call, "The number of imputations must be at least 2")
}
.meta.nsue(x, model, hypothesis, n.imp, n.bins, maxiter, 
tol)
}
metabias <-
function (x, ...) 
UseMethod("metabias")
metabias.meta.nsue <-
function (x, maxiter = 100, tol = 1e-06, ...) 
{
if (!inherits(x, "meta.nsue")) {
.stop(match.call(), "The argument must be a 'meta.nsue' object")
}
if (nrow(x$rm$M) < length(x$known$i) + length(x$unknown$i)) {
.warning("This analysis does not take repeated measures into account")
}
measure <- x$measure
known <- x$known$i
unknown <- x$unknown$i
mi_y = x$unknown$y
X <- x$model$matrix
n.coef_j <- ncol(x$model$matrix) + 1
y <- c()
y[known] <- x$known$y
se_coef <- c()
se_coef_var <- c()
for (j in 1:max(c(1, ncol(mi_y)))) {
if (ncol(mi_y) > 0) {
y[unknown] <- mi_y[, j]
}
if (measure == "cor" || measure == "cor in smd") {
y.var <- x$y.var
}
if (measure == "smc" || measure == "smd") {
y.var <- x$y2var_k1 + x$y2var_k2 * y^2
}
X_j <- cbind(X, sqrt(y.var))
W <- diag(1/(y.var + .tau2.reml(y, y.var, X_j, maxiter, 
tol)))
inv_XtWX <- solve(t(X_j) %*% W %*% X_j)
se_coef <- c(se_coef, (inv_XtWX %*% t(X_j) %*% W %*% 
y)[n.coef_j])
se_coef_var <- c(se_coef_var, diag(inv_XtWX)[n.coef_j])
}
z <- mean(se_coef)/sqrt(.pool.var(se_coef, se_coef_var))
names(z) <- "z"
p <- 2 * pnorm(-abs(z))
x <- list(method = "'meta.nsue' regression test for funnel plot asymmetry", 
data.name = as.character(match.call()[2]), statistic = z, 
p.value = p)
class(x) <- "htest"
x
}
plot.meta.nsue <-
function (x, ...) 
{
if (!inherits(x, "meta.nsue")) {
.stop(match.call(), "The argument must be a 'meta.nsue' object")
}
forest.meta.nsue(x)
}
print.leave1out.nsue <-
function (x, ...) 
{
if (!inherits(x, "leave1out.nsue")) {
.stop(match.call(), "The argument must be a 'leave1out.nsue' object")
}
cat("\n")
cat("Meta-analysis description:\n")
cat("- Measure:", .format.measure(x[[1]]$meta.nsue$measure), 
"\n")
cat("- Model: measure", x[[1]]$meta.nsue$model$formula, "\n")
cat("- Hypothesis: ", paste(x$hypothesis$text, collapse = " & "), 
"\n")
cat("\n")
for (i in 1:length(x)) {
cat("\n")
cat("Discarded study:", x[[i]]$study, "\n")
cat("\n")
.print.heterogeneity(x[[i]]$meta.nsue)
cat("\n")
.print.model(x[[i]]$meta.nsue)
cat("\n")
.print.hypothesis(x[[i]]$meta.nsue)
cat("\n")
}
.print.sign()
cat("\n")
}
print.meta.nsue <-
function (x, ...) 
{
if (!inherits(x, "meta.nsue")) {
.stop(match.call(), "The argument must be a 'meta.nsue' object")
}
cat("\n")
cat("Meta-analysis description:\n")
cat("- Measure:", .format.measure(x$measure), "\n")
cat("- Known measures:", length(x$known$i), "\n")
if (length(x$known$i) == 0) {
.warning("No known measures!")
}
cat("- Non-statistically-significant unknown measures:", 
length(x$unknown$i), "\n")
rm.n.stud = nrow(x$rm$M)
if (rm.n.stud < length(x$known$i) + length(x$unknown$i)) {
cat("- Measures after combination of repeated-measures:", 
rm.n.stud, "\n")
}
cat("- Imputations:", ncol(x$unknown$y), "\n")
cat("- Model: measure", x$model$formula, "\n")
cat("- Hypothesis: ", paste(x$hypothesis$text, collapse = " & "), 
"\n")
cat("\n")
.print.heterogeneity(x)
cat("\n")
.print.model(x)
cat("\n")
.print.hypothesis(x)
cat("\n")
.print.sign()
cat("\n")
}
print.nsue <-
function (x, ...) 
{
cat("\n")
cat("'nsue' object description:\n")
cat("- Measure:", .format.measure(x$measure), "\n")
known.n.stud = sum(!is.na(x$y))
unknown.n.stud = sum(is.na(x$y))
cat("- Known measures:", known.n.stud, "\n")
if (known.n.stud == 0) {
.warning("No known measures!")
}
cat("- Non-statistically-significant unknown measures:", 
sum(is.na(x$y)), "\n")
cat("\n")
}
r_in_smd_from_t_means_and_sds1 <-
function (t, n1, mean1.pre, sd1.pre, mean1.post, sd1.post, n2, 
mean2.pre, sd2.pre, mean2.post, sd2.post, alpha = 0.05, labels = "study", 
r.range = c(0, 0.99), rm.r = 0.3) 
{
call <- match.call()
if (missing(t) || missing(n1) || missing(n2)) {
.stop(call, "You must specify t, n1, mean1.pre, sd1.pre, mean1.post, sd1.post, n2, mean2.pre, sd2.pre, mean2.post and sd2.post")
}
if (!is.numeric(t)) {
.stop(call, "t is not a numeric vector")
}
n.stud <- length(t)
if (!n.stud) {
.stop(call, "No studies to meta-analyze")
}
n1 <- .check.n(call, n1, 2, n.stud)
if (!is.numeric(mean1.pre)) {
.stop(call, "mean1.pre is not a numeric vector")
}
if (!is.numeric(sd1.pre) || any(sd1.pre < 0)) {
.stop(call, "sd1.pre is not a positive numeric vector")
}
if (!is.numeric(mean1.post)) {
.stop(call, "mean1.post is not a numeric vector")
}
if (!is.numeric(sd1.post) || any(sd1.post < 0)) {
.stop(call, "sd1.post is not a positive numeric vector")
}
n2 <- .check.n(call, n2, 2, n.stud)
if (!is.numeric(mean2.pre)) {
.stop(call, "mean2.pre is not a numeric vector")
}
if (!is.numeric(sd2.pre) || any(sd2.pre < 0)) {
.stop(call, "sd2.pre is not a positive numeric vector")
}
if (!is.numeric(mean2.post)) {
.stop(call, "mean2.post is not a numeric vector")
}
if (!is.numeric(sd2.post) || any(sd2.post < 0)) {
.stop(call, "sd2.post is not a positive numeric vector")
}
alpha <- .check.alpha(call, alpha, n.stud)
labels <- .check.labels(call, labels, n.stud)
if (!is.numeric(r.range) || r.range[1] > r.range[2] || r.range[1] < 
-1 || r.range[2] > 1) {
.stop(call, "Incorrect r.range")
}
if (!is.numeric(rm.r) || rm.r < -1 || rm.r > 1) {
.stop(call, "Incorrect rm.r")
}
for (i in 1:n.stud) {
if ((is.na(t[i]) && is.na(alpha[i])) || is.na(n1[i]) || 
is.na(n2[i])) {
.stop("Not enough information in study", labels[i])
}
}
n <- n1 + n2
df <- n - 2
inv_n1_n2 <- 1/n1 + 1/n2
j <- .d_j(df)
df1 <- n1 - 1
df2 <- n2 - 1
diff <- mean1.post - mean1.pre - mean2.post + mean2.pre
k_t2var <- diff^2/inv_n1_n2
var1.sum <- sd1.pre^2 + sd1.post^2
var2.sum <- sd2.pre^2 + sd2.post^2
sd1.prod <- sd1.pre * sd1.post
sd2.prod <- sd2.pre * sd2.post
r.min <- r.range[1]
r.max <- r.range[2]
obj <- list(measure = "cor in smd", y = atanh(.r_in_smd_from_sds(k_t2var/t^2, 
df1, var1.sum, sd1.prod, df2, var2.sum, sd2.prod, r.min, 
r.max)), y_lo = atanh(rep(r.min, n.stud)), y_up = atanh(rep(r.max, 
n.stud)), y.var = (n2/n)^2/(n1 - 3) + (n1/n)^2/(n2 - 
3), smd = data.frame(diff, df1, var1.sum, sd1.prod, df2, 
var2.sum, sd2.prod, j = j, y2var_k1 = inv_n1_n2, y2var_k2 = 1 - 
(df - 2)/(df * j^2)), labels = labels, rm = list(r = rm.r))
class(obj) <- "nsue"
obj
}
r_in_smd_from_t_means_and_sds2 <-
function (x, formula = ~1, hypothesis = NULL, maxiter = 200, 
tol = 1e-06) 
{
call <- match.call()
is_meta = inherits(x, "meta.nsue")
is_leave1out = inherits(x, "leave1out.nsue")
if (!is_meta && !is_leave1out) {
.stop(call, "The argument must be a 'meta.nsue' or 'leave1out.nsue' object")
}
if (is_leave1out) {
n.stud = length(x[[1]]$meta.nsue$known$i) + length(x[[1]]$meta.nsue$unknown$i)
}
else {
n.stud = length(x$known$i) + length(x$unknown$i)
}
model <- .check.formula(call, formula, n.stud)
hypothesis <- .check.hypothesis(call, hypothesis, model)
if (is_leave1out) {
for (i in 1:length(x)) {
x[[i]]$meta.nsue = .r_in_smd_from_t_means_and_sds2(x[[i]]$meta.nsue, 
model, hypothesis, maxiter, tol)
}
return(x)
}
.r_in_smd_from_t_means_and_sds2(x, model, hypothesis, maxiter, 
tol)
}
residuals.meta.nsue <-
function (object, ...) 
{
fitted <- fitted(object)
known.i <- object$known$i
residuals <- c()
residuals[known.i] <- object$known$y - fitted[known.i]
unknown.i <- object$unknown$i
if (length(object$unknown$i) > 0) {
residuals[unknown.i] <- apply(object$unknown$y, 1, mean) - 
fitted[unknown.i]
}
residuals
}
smc_from_t <-
function (t, n, alpha = 0.05, labels = "study", rm.r = 0.3) 
{
call <- match.call()
if (missing(t) || missing(n)) {
.stop(call, "You must specify t and n")
}
if (!is.numeric(t)) {
.stop(call, "t is not a numeric vector")
}
n.stud <- length(t)
if (!n.stud) {
.stop(call, "No studies to meta-analyze")
}
n <- .check.n(call, n, 3, n.stud)
alpha <- .check.alpha(call, alpha, n.stud)
labels <- .check.labels(call, labels, n.stud)
if (!is.numeric(rm.r) || rm.r < -1 || rm.r > 1) {
.stop(call, "Incorrect rm.r")
}
for (i in 1:n.stud) {
if ((is.na(t[i]) && is.na(alpha[i])) || is.na(n[i])) {
stop("Not enough information in study", labels[i])
}
}
df <- n - 1
inv_n <- 1/n
j <- .d_j(df)
k_t2d <- j * sqrt(inv_n)
y_up <- k_t2d * qt(1 - alpha/2, df)
obj <- list(measure = "smc", y = k_t2d * t, y_lo = -y_up, 
y_up = y_up, y2var_k1 = inv_n, y2var_k2 = 1 - (df - 2)/(df * 
j^2), labels = labels, rm = list(r = rm.r))
class(obj) <- "nsue"
obj
}
smd_from_t <-
function (t, n1, n2, alpha = 0.05, labels = "study", rm.r = 0.3) 
{
call <- match.call()
if (missing(t) || missing(n1) || missing(n2)) {
.stop(call, "You must specify t, n1 and n2")
}
if (!is.numeric(t)) {
.stop(call, "t is not a numeric vector")
}
n.stud <- length(t)
if (!n.stud) {
.stop(call, "No studies to meta-analyze")
}
n1 <- .check.n(call, n1, 2, n.stud)
n2 <- .check.n(call, n2, 2, n.stud)
alpha <- .check.alpha(call, alpha, n.stud)
labels <- .check.labels(call, labels, n.stud)
if (!is.numeric(rm.r) || rm.r < -1 || rm.r > 1) {
.stop(call, "Incorrect rm.r")
}
for (i in 1:n.stud) {
if ((is.na(t[i]) && is.na(alpha[i])) || is.na(n1[i]) || 
is.na(n2[i])) {
stop("Not enough information in study", labels[i])
}
}
n <- n1 + n2
df <- n - 2
inv_n1_n2 <- 1/n1 + 1/n2
j <- .d_j(df)
k_t2d <- j * sqrt(inv_n1_n2)
y_up = k_t2d * qt(1 - alpha/2, df)
obj <- list(measure = "smd", y = k_t2d * t, y_lo = -y_up, 
y_up = y_up, y2var_k1 = inv_n1_n2, y2var_k2 = 1 - (df - 
2)/(df * j^2), labels = labels, rm = list(r = rm.r))
class(obj) <- "nsue"
obj
}
subset.nsue <-
function (x, subset, ...) 
{
call <- match.call()
if (!inherits(x, "nsue")) {
.stop(call, "The argument must be a 'nsue' object")
}
if (!is.logical(subset)) {
.stop(call, "subset must be logical")
}
if (length(subset) != length(x$y)) {
.stop(call, "wrong length")
}
measure <- x$measure
selected = which(subset)
x$y <- x$y[selected]
x$y_lo <- x$y_lo[selected]
x$y_up <- x$y_up[selected]
if (measure == "cor" || measure == "cor in smd") {
x$y.var <- x$y.var[selected]
}
if (measure == "smc" || measure == "smd") {
x$y2var_k1 <- x$y2var_k1[selected]
x$y2var_k2 <- x$y2var_k2[selected]
}
x$labels <- x$labels[selected]
if (measure == "cor in smd") {
x$smd <- x$smd[selected, ]
}
x
}
summary.leave1out.nsue <-
function (object, ...) 
{
if (!inherits(object, "leave1out.nsue")) {
.stop(match.call(), "The argument must be a 'leave1out.nsue' object")
}
cat("\n")
cat("Meta-analysis model:", object[[1]]$meta.nsue$measure, 
object[[1]]$meta.nsue$model$formula, "\n")
cat("\n")
for (i in 1:length(object)) {
cat("Discarded study:", object[[i]]$study, "\n")
.print.hypothesis(object[[i]]$meta.nsue)
cat("\n")
}
.print.sign()
cat("\n")
invisible(object)
}
summary.meta.nsue <-
function (object, ...) 
{
if (!inherits(object, "meta.nsue")) {
.stop(match.call(), "The argument must be a 'meta.nsue' object")
}
cat("\n")
cat("Meta-analysis model:", object$measure, object$model$formula, 
"\n")
cat("\n")
.print.hypothesis(object)
cat("\n")
.print.sign()
cat("\n")
invisible(object)
}
z_from_r <-
function (r, n, alpha = 0.05, labels = "study", rm.r = 0.3) 
{
call <- match.call()
if (missing(r) || missing(n)) {
.stop(call, "You must specify r and n")
}
if (!is.numeric(r)) {
.stop(call, "r is not a numeric vector")
}
if (any(r < -1, na.rm = TRUE) || any(r > 1, na.rm = TRUE)) {
.stop(call, "r cannot be <= -1 or >= 1")
}
n.stud <- length(r)
if (!n.stud) {
.stop(call, "No studies to meta-analyze")
}
n <- .check.n(call, n, 4, n.stud)
alpha <- .check.alpha(call, alpha, n.stud)
labels <- .check.labels(call, labels, n.stud)
if (!is.numeric(rm.r) || rm.r < -1 || rm.r > 1) {
.stop(call, "Incorrect rm.r")
}
for (i in 1:n.stud) {
if ((is.na(r[i]) && is.na(alpha[i])) || is.na(n[i])) {
stop("Not enough information in study", labels[i])
}
}
y_up = atanh((1 + (n - 2)/qt(alpha/2, n - 2)^2)^-0.5)
obj <- list(measure = "cor", y = atanh(r), y_lo = -y_up, 
y_up = y_up, y.var = 1/(n - 3), labels = labels, rm = list(r = rm.r))
class(obj) <- "nsue"
obj
}
.gui()
